#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging

log = logging.getLogger("terra.core.notify_list")


class NotifyList(list):
    def __init__(self, *args, **kargs):
        list.__init__(self, *args, **kargs)
        self.callback_changed = None
        self._frozen = 0
        self._dirty = False

    def freeze(self):
        self._frozen += 1

    def thaw(self):
        if self._frozen < 1:
            log.debug_error("Too much thaw")
            return

        self._frozen -= 1
        if self._frozen == 0 and self._dirty:
            if self.callback_changed is not None:
                self.callback_changed()
            self._dirty = False

    def notify_method(meth):
        def wrap(self, *args, **kargs):
            ret = meth(self, *args, **kargs)

            if self._frozen == 0:
                if self.callback_changed is not None:
                    self.callback_changed()
            else:
                self._dirty = True

            return ret

        return wrap

    __delitem__ = notify_method(getattr(list, "__delitem__"))
    __setitem__ = notify_method(getattr(list, "__setitem__"))
    __delslice__ = notify_method(getattr(list, "__delslice__"))
    __setslice__ = notify_method(getattr(list, "__setslice__"))
    append = notify_method(getattr(list, "append"))
    pop = notify_method(getattr(list, "pop"))
    extend = notify_method(getattr(list, "extend"))
    insert = notify_method(getattr(list, "insert"))
    remove = notify_method(getattr(list, "remove"))
    reverse = notify_method(getattr(list, "reverse"))
    sort = notify_method(getattr(list, "sort"))
