#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging
from grid import BaseGrid
from kinetic import KineticMouse

__all__ = ("BaseVerticalGrid", "KineticVerticalGrid")

log = logging.getLogger("terra.ui.vertical_grid")


class BaseVerticalGrid(BaseGrid):
    """Basic grid of items that scrolls vertically.

    See BaseGrid for more details.
    @see: L{BaseGrid}
    """
    def __init__(self, canvas, renderer_new, elements, renderer_size=(32, 32),
                 h_align=0.5):
        self.w_items = 0
        self.items_per_col = 0
        BaseGrid.__init__(self, canvas, renderer_new, elements, renderer_size)
        self.offset = -renderer_size[1]
        self.h_align = h_align
        self.inc_x = 0
        self.offset_x = 0

    def position_set(self, value):
        if value < 0.0:
            log.warning("value (%s) < 0.0, set to 0.0", value)
            value = 0.0
        elif value > 1.0:
            log.warning("value (%s) > 1.0, set to 1.0", value)
            value = 1.0

        max_y = self.last_top_left_visible * self.renderer_height
        y = int(max_y * value)
        self.current = y / self.renderer_height
        self.offset = -self.renderer_height - (y % self.renderer_height)
        self._reposition_renderers()
        self._refill_renderers()
        self.emit_position_changed()

    def position_get(self):
        "Get position in grid, from 0.0 (top) to 1.0 (bottom)."
        if self.last_top_left_visible < 1:
            return 0.0
        else:
            rel_offset = self.offset + self.renderer_height
            pos = self.current + float(rel_offset) / -self.renderer_height
            return float(pos) / self.last_top_left_visible

    def visible_rows_count(self):
        if not self.renderers:
            return 0
        r = len(self.renderers)
        e = len(self.elements)
        if r == e:
            return e - 1
        elif r > e:
            return e
        else:
            return max(r - self.spare_renderers_rows, 0)

    def visible_cols_count(self):
        return self.w_items

    def visible_rows_scale_get(self):
        v = self.visible_rows_count()
        e = self.items_per_col
        if v >= e or e == 0:
            return 1.0
        else:
            return float(v) / e

    def renderers_get(self):
        result = []
        for row in self.renderers:
            for c in row:
                result.append(c)
        return result

    def renderer_for_index(self, index):
        if self.current is None:
            return None
        if not (0 <= index < len(self.elements)):
            return None

        col_idx = index % self.items_per_row
        row_idx = (index / self.items_per_row)
        row_idx -= (self.current - 1)
        if 0 <= row_idx <= len(self.renderers):
            return self.renderers[row_idx][col_idx]

    def _index_at_xy(self, x, y):
        ox, oy = self.pos
        x -= ox
        y -= oy + self.offset

        col_idx = x / self.renderer_width
        row_idx = y / self.renderer_height

        if col_idx >= self.items_per_row:
            return -1

        idx = (row_idx + self.current - 1) * self.items_per_row + col_idx

        if 0 <= idx < len(self.elements):
            return idx
        else:
            return -1

    def _reconfigure_renderers(self, x, y, w, h):
        if not self._dirty:
            return

        c_width, c_height = self.renderer_size
        self.w_items = w_items = max(int(w / c_width), 1)
        self.h_items = h_items = max(int(h / c_height), 1)
        self.n_items = n_items = w_items * h_items

        if self.h_align < 0 and w_items > 1:
            self.offset_x = 0
            self.inc_x = int((w - w_items * c_width) / (w_items - 1))
        else:
            self.offset_x = int(self.h_align * (w - w_items * c_width))
            self.inc_x = 0

        n_elements = len(self.elements)
        if n_items >= n_elements:
            self.current = 0
            self.offset = -c_height

        self.items_per_row = w_items
        if n_items >= n_elements:
            self.items_per_col = h_items
            self.last_top_left_visible = 0
        else:
            self.items_per_col = n_elements / w_items
            if n_elements % w_items:
                self.items_per_col += 1
            self.last_top_left_visible = self.items_per_col - h_items

        if h % c_height:
            self.spare_renderers_rows = 3
        else:
            self.spare_renderers_rows = 2
        h_items += self.spare_renderers_rows

        if self.last_top_left_visible < 0:
            self.last_top_left_visible = 0

        # delete unneeded
        if len(self.renderers):
            # delete unrequired rows of renderers
            while h_items < len(self.renderers):
                for r in self.renderers.pop():
                    r.delete()

            # delete unrequired columns of renderers
            if len(self.renderers) and w_items < len(self.renderers[0]):
                while w_items < len(self.renderers[0]):
                    for row in self.renderers:
                        cell = row.pop()
                        cell.delete()

        # reposition existing
        cy = y + self.offset
        for row in self.renderers:
            cx = x + self.offset_x
            for c in row:
                c.move(cx, cy)
                cx += c_width + self.inc_x
            cy += c_height

        # create required columns of renderers
        if len(self.renderers):
            if w_items > len(self.renderers[0]):
                cx = x + len(self.renderers[0]) * c_width + self.offset_x
                while w_items > len(self.renderers[0]):
                    cy = y + self.offset
                    for row in self.renderers:
                        c = self._renderer_new_get()
                        c.geometry_set(cx, cy, c_width, c_height)
                        row.append(c)
                        cy += c_height
                    cx += c_width + self.inc_x

        # create required rows of renderers
        cy = y + len(self.renderers) * c_height + self.offset
        while h_items > len(self.renderers):
            cx = x + self.offset_x
            row = []
            for i in xrange(w_items):
                c = self._renderer_new_get()
                c.geometry_set(cx, cy, c_width, c_height)
                row.append(c)
                cx += c_width + self.inc_x
            self.renderers.append(row)
            cy += c_height

        self._refill_renderers()
        self._dirty = False

    def _hide_all_renderers(self):
        for row in self.renderers:
            for c in row:
                c.hide()

    def _hide_top_spare_row(self):
        self._apply_to_renders_row(0, lambda c: c.hide())

    def _hide_bottom_spare_rows(self):
        def hide(c):
            c.hide()
        for i in xrange(-self.spare_renderers_rows + 1, 0):
            self._apply_to_renders_row(i, hide)

    def _hide_spare_rows(self):
        self._hide_top_spare_row()
        self._hide_bottom_spare_rows()

    def _refill_renderers_row(self, row, idx):
        n_elements = len(self.elements)
        for c in row:
            if idx < n_elements:
                c.value_set(self.elements[idx])
                c.show()
            else:
                c.hide()
            idx += 1

    def _refill_renderers(self):
        if not self.renderers or not self.renderers[0]:
            return

        n_elements = len(self.elements)
        if n_elements < 1 or self.current is None:
            return self._hide_all_renderers()

        if self.current == 0:
            self._hide_top_spare_row()
            row_idx_start = 1
        else:
            row_idx_start = 0

        if self.current == self.last_top_left_visible:
            row_idx_end = self.h_items + 1
            self._hide_bottom_spare_rows()
        else:
            row_idx_end = len(self.renderers)
            if self.current == self.last_top_left_visible - 1 \
               and self.spare_renderers_rows > 2:
                row_idx_end -= 1
                self._apply_to_renders_row(-1, lambda c: c.hide())

        idx = (self.current - 1 + row_idx_start) * self.items_per_row
        for row_idx in xrange(row_idx_start, row_idx_end):
            row = self.renderers[row_idx]
            self._refill_renderers_row(row, idx)
            idx += self.items_per_row

    def _reposition_renderers(self):
        if self.offset == self.last_offset:
            return

        self.last_offset = self.offset
        x, y = self.pos
        cy = y + self.offset
        c_width, c_height = self.renderer_size
        for row in self.renderers:
            cx = x + self.offset_x
            for c in row:
                c.move(cx, cy)
                cx += c_width + self.inc_x
            cy += c_height

    def move_offset(self, offset):
        if offset == 0 or self.current is None \
               or not self.renderers or not self.renderers[0] \
               or len(self.elements) <= self.n_items:
            return False

        max_y = self.last_top_left_visible * self.renderer_height
        old_y = (self.current - 1) * self.renderer_height - self.offset
        y = old_y - offset

        if y < 0 or y > max_y:
            if y < 0:
                self.current = 0
            else:
                self.current = self.last_top_left_visible
            self.offset = -self.renderer_height
            self._refill_renderers()
            self._reposition_renderers()
            self.emit_position_changed()
            return False

        idx = y / self.renderer_height
        items_over = idx - self.current
        if items_over == 0:
            self.offset += offset
        elif items_over in (-1, 1):
            self.current += items_over
            self.offset = (self.current - 1) * self.renderer_height - y
            if items_over == 1:
                renderer = self.renderers.pop(0)
                self.renderers.append(renderer)
                new_idx = self.current + len(self.renderers) - 2
            else:
                renderer = self.renderers.pop()
                self.renderers.insert(0, renderer)
                new_idx = self.current - 1

            if 0 <= new_idx < self.items_per_col:
                n = new_idx * self.items_per_row
                self._refill_renderers_row(renderer, n)
            else:
                for c in renderer:
                    c.hide()

        else:
            self.current += items_over
            self.offset = (self.current - 1) * self.renderer_height - y
            self._refill_renderers()

        self._reposition_renderers()
        self.emit_position_changed()
        return True



class KineticVerticalGrid(BaseVerticalGrid):
    """Vertical Grid using kinetics.

    Events will not be handled to row renderers, they will be handled by
    event_area, an object that will be laid out on top of every renderer.

    Behavior is based on B{click_constant}: maximum vertical motion to
    consider as click, values greater that (in absolute terms) are considered
    drag.
    """
    def __init__(self, canvas, renderer_new, elements, renderer_size=(32, 32),
                 h_align=0.5):
        BaseVerticalGrid.__init__(self, canvas, renderer_new,
                                  elements, renderer_size, h_align)
        self.event_area = self.Rectangle(color=(0, 0, 0, 0))
        self.event_area.show()
        self.event_area.on_mouse_down_add(self._cb_on_mouse_down)
        self.event_area.on_mouse_up_add(self._cb_on_mouse_up)
        self.event_area.on_mouse_move_add(self._cb_on_mouse_move)
        self.kinetic = KineticMouse(self.move_offset)
        self.is_drag = False
        self.mouse_down_pos = None

    def _renderer_new_get(self):
        o = BaseVerticalGrid._renderer_new_get(self)
        self.event_area.raise_()
        return o

    def resize(self, w, h):
        BaseVerticalGrid.resize(self, w, h)
        self.event_area.resize(w, h)

    def _cb_on_mouse_down(self, obj, event):
        if event.button == 1:
            y = event.position.canvas.y
            self.mouse_down_pos = y
            self.is_drag = not self.kinetic.mouse_down(y)

    def _is_click_possible(self, y):
        if self.is_drag or self.mouse_down_pos is None:
            return False
        else:
            return abs(y - self.mouse_down_pos) <= self.click_constant

    def _cb_on_mouse_up(self, obj, event):
        if event.button == 1:
            x = event.position.canvas.x
            y = event.position.canvas.y
            if self._is_click_possible(y):
                self.emit_clicked(x, y)
                self.kinetic.mouse_cancel()
            else:
                self.kinetic.mouse_up(y)

    def _cb_on_mouse_move(self, obj, event):
        if event.buttons == 1:
            y = event.position.canvas.y
            if not self._is_click_possible(y):
                self.is_drag = True
            if abs(self.mouse_down_pos - y) > self.move_constant:
                self.kinetic.mouse_move(y)
